"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineConfigurationRoutes = defineConfigurationRoutes;
exports.deleteConfigCache = deleteConfigCache;
exports.deleteConfigResource = deleteConfigResource;
exports.getConfigAliases = getConfigAliases;
exports.getConfigIndexMappings = getConfigIndexMappings;
exports.getConfigIndices = getConfigIndices;
exports.getConfigResourceById = getConfigResourceById;
exports.getConfigResourcesByType = getConfigResourcesByType;
exports.getRestApiInfo = getRestApiInfo;
exports.updateConfigResource = updateConfigResource;
var _configSchema = require("@kbn/config-schema");
var _wrap_elasticsearch_error = require("../../../../utils/wrap_elasticsearch_error");
var _constants = require("../../../../utils/constants");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

function getConfigResourcesByType({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const results = await searchGuardConfigurationBackend.list(request.headers, request.params.resourceName);
      return response.ok({
        body: {
          total: Object.keys(results).length,
          data: results
        }
      });
    } catch (error) {
      logger.error(`getConfigResourcesByType: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigResourceById({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardConfigurationBackend.get(request.headers, request.params.resourceName, request.params.id);
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigResourceById: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function deleteConfigResource({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const result = await searchGuardConfigurationBackend.delete(request.headers, request.params.resourceName, request.params.id);
      return response.ok({
        body: {
          message: result.message
        }
      });
    } catch (error) {
      logger.error(`deleteConfigResource: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function updateConfigResource({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const result = await searchGuardConfigurationBackend.save(request.headers, request.params.resourceName, request.params.id, request.body);
      return response.ok({
        body: {
          message: result.message
        }
      });
    } catch (error) {
      logger.error(`updateConfigResource: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function deleteConfigCache({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const result = await searchGuardConfigurationBackend.clearCache(request.headers);
      return response.ok({
        body: {
          message: result.message
        }
      });
    } catch (error) {
      logger.error(`deleteConfigCache: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getRestApiInfo({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardConfigurationBackend.restapiinfo(request.headers);
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getRestApiInfo: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigIndices({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const {
        headers,
        body: {
          index
        }
      } = request;
      const body = await searchGuardConfigurationBackend.indices({
        headers,
        index
      });
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigIndices: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigAliases({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const {
        headers,
        body: {
          alias
        }
      } = request;
      const body = await searchGuardConfigurationBackend.aliases({
        headers,
        alias
      });
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigAliases: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigIndexMappings({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardConfigurationBackend.getIndexMappings({
        headers: request.headers,
        body: request.body
      });
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigIndexMappings: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}

/**
 * The backend API allows to manage the backend configuration.
 */
function defineConfigurationRoutes({
  searchGuardConfigurationBackend,
  kibanaCore,
  logger
}) {
  const router = kibanaCore.http.createRouter();

  /**
   * Returns a list of resource instances.
   *
   */
  router.get({
    path: `${_constants.API_ROOT}/configuration/{resourceName}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigResourcesByType({
    searchGuardConfigurationBackend,
    logger
  }));

  /**
   * Returns a resource instance.
   *
   * Response sample:
   *
   * {
   *   "id": "kibiuser",
   * }
   */
  router.get({
    path: `${_constants.API_ROOT}/configuration/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigResourceById({
    searchGuardConfigurationBackend,
    logger
  }));

  /**
   * Deletes a resource instance.
   *
   * Response sample:
   *
   * {
   *   "message": "Deleted user username"
   * }
   */
  router.delete({
    path: `${_constants.API_ROOT}/configuration/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, deleteConfigResource({
    searchGuardConfigurationBackend,
    logger
  }));

  /**
   * Updates or creates a resource
   *
   * Request sample:
   *
   * {
   *   "password": "password"
   * }
   */
  router.post({
    path: `${_constants.API_ROOT}/configuration/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({}, {
        unknowns: 'allow'
      })
    },
    options: {
      authRequired: true
    }
  }, updateConfigResource({
    searchGuardConfigurationBackend,
    logger
  }));
  router.delete({
    path: `${_constants.API_ROOT}/configuration/cache`,
    validate: false,
    options: {
      authRequired: true
    }
  }, deleteConfigCache({
    searchGuardConfigurationBackend,
    logger
  }));
  router.get({
    path: `${_constants.API_ROOT}/restapiinfo`,
    validate: false,
    options: {
      authRequired: true
    }
  }, getRestApiInfo({
    searchGuardConfigurationBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/configuration/indices`,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigIndices({
    searchGuardConfigurationBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/configuration/aliases`,
    validate: {
      body: _configSchema.schema.object({
        alias: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigAliases({
    searchGuardConfigurationBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/configuration/index_mappings`,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigIndexMappings({
    searchGuardConfigurationBackend,
    logger
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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